﻿/*============================================================================
                                                                    2007.01.02
  FINAL FANTASY XI ONLINE 支援ツール
  えふえふじゅういち時間 Flash版 ～ VanaDate クラス ver1.02
 
                                             Copyright (c) 2005-2007 by koji27
                                                http://www.koji27.com/soft/ft/
==============================================================================
要素
------------------------------------------------------------------------------

このクラスを使用すると、ヴァナディール時間を調べることができます。基本的に 
ActionScript 2.0 の Date クラスと同等なメソッドを持ちますが、以下の部分が異な
ります。

1.	set系のメソッドの代わりに、メソッドで指定された単位でヴァナ時間をインクリ
		メントするinc系のメソッドを持つ。
2.	getMonth、setMonth メソッドは1～12でヴァナ月を扱う。(Date クラスは0～11)
3.	ActionScript 2.0 では複数のコンストラクタを持つクラスを作ることができない
		ため、インスタンスを作成したときの初期値は作成時のヴァナ時間になる。

また、Date クラスは地球時間をミリ秒単位でも返しますが、VanaDate クラスが扱うヴ
ァナ時間の最小単位は秒です。ちなみに、ヴァナ時間は地球時間の25倍の早さで進んで
います。ヴァナ秒は地球時間の1/25秒(40ミリ秒)に相当します。

ただし、英語版のFF11は月齢の表示が日本語版とはちょっと違います。このクラスでは
英語版の月齢表記に完全には対応していません。

==============================================================================
プロパティ・メソッド
------------------------------------------------------------------------------

・スタティック プロパティ
  
	DAY_JP    - 日本語のヴァナ曜日配列
	DAY_EN    - 英語のヴァナ曜日配列
	MOON_JP   - 日本語のヴァナ月齢配列
	MOON_EN   - 英語のヴァナ月齢配列
	
・パブリックメソッド
	getYear			- ヴァナ年を返します。(1～999)
	getMonth		- ヴァナ月を返します。(1～12)
	getDate			- ヴァナ日を返します。(1～30)
	getHours		- ヴァナ時を返します。(0～23)
	getMinutes	- ヴァナ分を返します。(0～59)
	getSeconds	- ヴァナ秒を返します。(0～59)
	getDay			- ヴァナ曜日を返します。(0～7)
	getMoon			- ヴァナ月齢を返します。(0～11)
	getTime			- 1年1月1日 0時0分0秒からの経過時間をヴァナ秒で返します。
	incYear			- 指定したヴァナ年でインクリメントします。(マイナスの指定も可能)
	incMonth		- 指定したヴァナ月でインクリメントします。(マイナスの指定も可能)
	incDate			- 指定したヴァナ日でインクリメントします。(マイナスの指定も可能)
	incHours		- 指定したヴァナ時でインクリメントします。(マイナスの指定も可能)
	incMinutes	- 指定したヴァナ分でインクリメントします。(マイナスの指定も可能)
	incSeconds	- 指定したヴァナ秒でインクリメントします。(マイナスの指定も可能)
	setTime			- 指定したヴァナ秒でヴァナ時間を初期化します。
	setDateTime	- 指定した日時でヴァナ時間を初期化します。
								DavaDate.setDateTime(year, month, date, hours, minutes, seconds);
	
	getUTC			- ヴァナ時間を地球時間(UTC ミリ秒)に変換して返します。
	getLocal		- ヴァナ時間を地球時間(ローカル ミリ秒)に変換して返します。
	setUTC			- 地球時間(UTC ミリ秒)でヴァナ時間を初期化します。
	setLocal		- 地球時間(ローカル ミリ秒)でヴァナ時間を初期化します。
	
	toString		- 0001/01/01 00:00:00 形式でヴァナ時間を返します。
	toStringJp	- 0001/01/01(火) 00:00:00 <満月> 形式でヴァナ時間を返します。
	toStringEn	- Firesday 1 1 00:00:00 1 <Full Moon> 形式でヴァナ時間を返
								します。

==============================================================================
著作権・免責事項など
------------------------------------------------------------------------------

VanaDate クラス (以下 このクラス) の著作権は koji27 こと安達恒司 (以下 作者) 
にあります。

作者はこのクラスのいかなる事項に関しても一切の保証をいたしません。

作者はこのクラスを使用することで生じた、いかなる損害に対しても、その損害が予測
可能であったかどうかに関わらず、一切の責任を負いません。

作者はこのクラスにバグ等の不具合が発見されても、不具合修正作業、及び不具合修正
版の配布等の一切の責任を負いません。

作者はこのクラスの再利用等に関して、細かく条件を付けたりはしません。

==============================================================================
更新履歴
------------------------------------------------------------------------------

var1.02 (2007.01.02)
・地球時間が1ヵ月ずれていた不具合を修正

var1.01 (2006.09.28)
・最初の公開バージョン

ver1.00 (2005.11.06)
・最初のバージョン(非公開)

============================================================================*/

class VanaDate {
	//--------------------------------------------------------------------------
	// プライベート プロパティ 
	private var vanaYear:Number;		// ヴァナ年
	private var vanaMonth:Number;		// ヴァナ月
	private var vanaDate:Number;		// ヴァナ日
	private var vanaHours:Number;		// ヴァナ時
	private var vanaMinutes:Number;	// ヴァナ分
	private var vanaSeconds:Number;	// ヴァナ秒
	private var	vanaDay:Number;			// ヴァナ曜日
	private var vanaMoon:Number;		// ヴァナ月齢
	
	private var vanaTime:Number;		// ヴァナ1年1月1日 0時0分0秒からの経過秒

	//--------------------------------------------------------------------------
	// パブリック スタティック プロパティ
	
	// ヴァナ曜日 日本語名
	public static var DAY_JP:Array = new Array(
			"火", "土", "水", "風", "氷", "雷", "光", "闇"
		);

	// ヴァナ曜日 英語名
	public static var DAY_EN:Array = new Array(
			"Firesday", "Earthsday", "Watersday", "Windsday", 
			"Iceday", "Lightningday", "Lightsday", "Darksday"
		);
	
	// ヴァナ月齢 日本語名
	public static var MOON_JP:Array = new Array(
			"二十日月", "二十六夜", "新月", "三日月", "七日月", "上弦の月", 
			"十日月", "十三夜", "満月", "十六夜", "居待月", "下弦の月"
		);

	// ヴァナ月齢 英語名 ※海外版では月齢の種類が日本語版とちょっと違う
	public static var MOON_EN:Array = new Array(
			"Waning Crescent", "Waning Crescent", "New Moon", 
			"Waxing Crescent", "Waxing Crescent", "First Quarter Moon", 
			"Waxing Gibbous", "Waxing Gibbous", "Full Moon", 
			"Waning Gibbous", "Waning Gibbous", "Last Quarter Moon"
		);
	
	//--------------------------------------------------------------------------
	// プライベート メソッド
	
	// UTC時間とローカル時間の差をミリ秒で取得
	private function getUTCOffset() {
		var d:Date = new Date();
		var n:Number;
		n = d.getTimezoneOffset();
		
		// 分をミリ秒に変換
		return n * 60 * 1000;
	}
	
	// 原点 地球時間(UTC)をミリ秒で取得
	private function getOriginUTC() {
		// 原点 地球時間 2003/12/01 18:00:01:500 UTC+09:00:00
    //           UTC 2003/12/01 09:00:01:500
		var d:Date = new Date(2003, 11, 1, 9, 0, 1, 500);
		return d.getTime();
	}
	
	// 原点 ヴァナ時間を秒で取得
	private function getOriginTime() {
		// 原点 ヴァナディール時間 0934/08/04 18:00:00
		var n:Number;
		
		n =
			934			* 360	* 24	* 60	* 60 +
			(8 - 1)	* 30	* 24	* 60	* 60 +
			(4 - 1)				* 24	* 60	* 60 +
			18									* 60	* 60;
//		+
//		0													* 60 +
//		0;

		return n;
	}
	
	// 地球時間(UTC ミリ秒)をヴァナ時間(秒)に変換
	private function UTCToVana(utc:Number):Number {
		var utcOrigin:Number	= this.getOriginUTC();
		var vanaOrigin:Number	= this.getOriginTime() * 1000;
		var vanaResult:Number;
		
		// ヴァナ時間を計算
		// 地球起点時間とutcの差を25倍して
		// ヴァナ起点時間に加えることで、ヴァナ現在時間(ミリ秒)を算出している。
		vanaResult = vanaOrigin + (utc - utcOrigin) * 25;
		
		// ヴァナミリ秒をヴァナ秒に変換して返す
		return Math.round(vanaResult / 1000);
	}
	
	// ヴァナ時間(秒)を地球時間(UTC ミリ秒)に変換
	public function VanaToUTC(vana:Number):Number {
		var utcOrigin:Number	= this.getOriginUTC();
		var vanaOrigin:Number	= this.getOriginTime() * 1000;
		var utcResult:Number;
		
		// 地球時間を計算
		utcResult = utcOrigin + (vana * 1000 - vanaOrigin) / 25;
		
		return utcResult;
	}

	//--------------------------------------------------------------------------
	// パブリック メソッド
	
	// ヴァナ年を返す 1～9999
	public function getYear():Number {
		return this.vanaYear;
	}
	
	// ヴァナ月を返す 1～12
	public function getMonth():Number {
		return this.vanaMonth;
	}
	
	// ヴァナ日を返す 1～30
	public function getDate():Number {
		return this.vanaDate;
	}
	
	// ヴァナ時を返す 0～23
	public function getHours():Number {
		return this.vanaHours;
	}
	
	// 分を返す 0～59
	public function getMinutes():Number {
		return this.vanaMinutes;
	}
	
	// ヴァナ秒を返す 0～59
	public function getSeconds():Number {
		return this.vanaSeconds;
	}
	
	// ヴァナ曜日を返す 0～7 (火～闇)
	public function getDay():Number {
		return this.vanaDay;
	}
	
	// ヴァナ月齢を返す 0～11 (二十日月～下弦の月)
	public function getMoon():Number {
		return this.vanaMoon;
	}
	
	// ヴァナ1年1月1日 0時0分0秒からの経過秒を返す
	public function getTime():Number {
		return this.vanaTime;
	}
	
	// 指定したヴァナ秒で初期化する
	public function setTime(seconds:Number):Number {
		var n:Number;
		if (this.vanaTime != seconds) {
			this.vanaTime = seconds;
			
			// 曜日
			n							= seconds					% (24 * 60 * 60);
			this.vanaDay	= ((seconds - n)	/ (24 * 60 * 60)) % 8;
			
			// 月齢
			n							= (seconds + 24 * 60 * 60 * 2)				% (24 * 60 * 60 * 7);
			this.vanaMoon	= (((seconds + 24 * 60 * 60 * 2) - n)	/ (24 * 60 * 60 * 7)) % 12;
			
			// 年
			n									= seconds 			% (360	* 24	* 60	* 60);
			this.vanaYear 		= (seconds - n) / (360	* 24	* 60	* 60);
			seconds						= n;
			
			// 月
			n									= seconds				% (30		* 24	* 60	* 60);
			this.vanaMonth		= (seconds - n)	/ (30		* 24	* 60	* 60) + 1;
			seconds						= n;
			
			// 日
			n									= seconds 			% (				24	* 60	* 60);
			this.vanaDate			= (seconds - n)	/ (				24	* 60	* 60) + 1;
			seconds						= n;
			
			// 時
			n									= seconds				% (							60	* 60);
			this.vanaHours		= (seconds - n)	/ (							60	* 60);
			seconds						= n;
			
			// 分
			n									= seconds				% (										60);
			this.vanaMinutes	= (seconds - n)	/ (										60);
			seconds						= n;
			
			// 秒
			this.vanaSeconds	= seconds;
		}
		return this.vanaTime;
	}
	
	// 指定したヴァナ日時で初期化する 成功したら ture 失敗したら false
	public function setDateTime(
		year:Number, month:Number, date:Number, 
		hours:Number, minutes:Number, seconds:Number):Boolean
	{
		var vanatime:Number;
		
		// 範囲チェック: 年は1～9999まで 月は1～12まで 日は1～30まで
		//               時は0～23まで   分は0～59まで 秒は0～59まで
		if ( 
				(year			< 1)	or (year		> 9999)	or 
				(month		< 1)	or (month		> 12)		or 
				(date			< 1)	or (date		> 30)		or
				(hours		< 0)	or (hours		> 23)		or 
				(minutes	< 0)	or (minutes	> 59)		or 
				(seconds	< 0)	or (seconds	> 59)
			)
		{
			// 範囲外なので false を返して終了
			return false;
		} else {
			// 範囲内なので処理
			vanatime = 
				year		* 360	* 24	* 60	* 60	+
				month		* 30	* 24	* 60	* 60	+
				date					* 24	* 60	* 60	+
				hours								* 60	* 60	+
				minutes										* 60	+
				seconds													;
			
			this.setTime(vanatime);
			return true;
		}
	}
	
	// ヴァナ年でインクリメントする
	public function incYear(year:Number):Number {
		var incseconds:Number = year * 360 * 24 * 60 * 60;
		return this.setTime(this.vanaTime + incseconds);
	}
	
	// ヴァナ月でインクリメントする
	public function incMonth(month:Number):Number {
		var incseconds:Number = month * 30 * 24 * 60 * 60;
		return this.setTime(this.vanaTime + incseconds);
	}
	
	// ヴァナ日でインクリメントする
	public function incDate(date:Number):Number {
		var incseconds:Number = date * 24 * 60 * 60;
		return this.setTime(this.vanaTime + incseconds);
	}
	
	// ヴァナ時でインクリメントする
	public function incHours(hours:Number):Number {
		var incseconds:Number = hours * 60 * 60;
		return this.setTime(this.vanaTime + incseconds);
	}

	// ヴァナ分でインクリメントする
	public function incMinutes(minutes:Number):Number {
		var incseconds:Number = minutes * 60;
		return this.setTime(this.vanaTime + incseconds);
	}
	
	// ヴァナ秒でインクリメントする
	public function incSeconds(seconds:Number):Number {
		return this.setTime(this.vanaTime + seconds);
	}
	
	// 0983/07/27 00:00:00 形式でヴァナ時間を返す
	public function toString():String {
		var year:String			= String(this.vanaYear);
		var month:String		= String(this.vanaMonth);
		var date:String			= String(this.vanaDate);
		var hours:String		= String(this.vanaHours);
		var minutes:String	= String(this.vanaMinutes);
		var seconds:String	= String(this.vanaSeconds);
		
		// 泥臭い処理
		if (year.length			== 1) year		= "000"	+ year;
		if (year.length			== 2) year		= "00"	+ year;
		if (year.length			== 3) year		= "0"		+ year;
		if (month.length		== 1) month		= "0"		+ month;
		if (date.length			== 1) date		= "0"		+ date;
		if (hours.length		== 1) hours		= "0"		+ hours;
		if (minutes.length	== 1) minutes	= "0"		+ minutes;
		if (seconds.length	== 1) seconds	= "0"		+ seconds;
		
		return year + "/" + month + "/" + date + " " + 
			hours + ":" + minutes + ":" + seconds; 
	}
	
	// 0001/01/01(火) 00:00:00 <満月> 形式でヴァナ時間を返す
	public function toStringJp():String {
		var year:String			= String(this.vanaYear);
		var month:String		= String(this.vanaMonth);
		var date:String			= String(this.vanaDate);
		var hours:String		= String(this.vanaHours);
		var minutes:String	= String(this.vanaMinutes);
		var seconds:String	= String(this.vanaSeconds);
		var day:String			= DAY_JP[this.vanaDay];
		var moon:String			= MOON_JP[this.vanaMoon];
		
		// 泥臭い処理
		if (year.length			== 1) year		= "000"	+ year;
		if (year.length			== 2) year		= "00"	+ year;
		if (year.length			== 3) year		= "0"		+ year;
		if (month.length		== 1) month		= "0"		+ month;
		if (date.length			== 1) date		= "0"		+ date;
		if (hours.length		== 1) hours		= "0"		+ hours;
		if (minutes.length	== 1) minutes	= "0"		+ minutes;
		if (seconds.length	== 1) seconds	= "0"		+ seconds;
		
		return year + "/" + month + "/" + date + "(" + day + ") " + 
			hours + ":" + minutes + ":" + seconds + " <" + moon + ">"; 
	}
	
	// Firesday 1 1 00:00:00 1 <Full Moon> 形式でヴァナ時間を返す
	public function toStringEn():String {
		var year:String			= String(this.vanaYear);
		var month:String		= String(this.vanaMonth);
		var date:String			= String(this.vanaDate);
		var hours:String		= String(this.vanaHours);
		var minutes:String	= String(this.vanaMinutes);
		var seconds:String	= String(this.vanaSeconds);
		var day:String			= DAY_EN[this.vanaDay];
		var moon:String			= MOON_EN[this.vanaMoon];
		
		// 泥臭い処理
		if (hours.length		== 1) hours		= "0"		+ hours;
		if (minutes.length	== 1) minutes	= "0"		+ minutes;
		if (seconds.length	== 1) seconds	= "0"		+ seconds;
		
		return day + " " + month + " " + date + " " + 
			hours + ":" + minutes + ":" + seconds + " " + year + " <" + moon + ">"; 
	}
	
	
	// 地球時間(UTC)のミリ秒を返す
	public function getUTC() {
		return this.VanaToUTC(this.vanaTime);
	}
	
	// 地球時間(ローカル)のミリ秒を返す
	public function getLocal() {
		return this.VanaToUTC(this.vanaTime) - this.getUTCOffset();
	}
		
	// 地球時間(UTC)のミリ秒でヴァナ時間を初期化し、
	// ヴァナ1年1月1日 0時0分0秒からの経過秒を返す
	public function setUTC(milliseconds:Number):Number {
		return this.setTime(this.UTCToVana(milliseconds));
	}
	
	public function setLocal(milliseconds:Number):Number {
		return setUTC(milliseconds + this.getUTCOffset());
	}
	
	//--------------------------------------------------------------------------
	// コンストラクタ
	public function VanaDate() {
		// 現在のヴァナ時間で初期化
		var d:Date = new Date();
		this.setLocal(d.getTime() - this.getUTCOffset());
	}
}
